// Basic frontend app using Fabric + Tesseract
const canvas = new fabric.Canvas('c', { selection: true, preserveObjectStacking: true });
let originalImage = null;
let ocrResults = [];

// Helpers to load image file into fabric canvas
function loadImageFile(file){
  const reader = new FileReader();
  reader.onload = function(e){
  document.getElementById("originalPreview").src = e.target.result;
    fabric.Image.fromURL(e.target.result, img => {
      const scale = Math.min(canvas.width / img.width, canvas.height / img.height, 1);
      img.set({ left: 0, top: 0, selectable: false, hasControls:false, hoverCursor:'default' });
      img.scale(scale);
      canvas.clear();
      canvas.setBackgroundImage(img, canvas.renderAll.bind(canvas));
      originalImage = img;
      ocrResults = [];
      document.getElementById('ocrBoxes').innerHTML = '';
    });
  }
  reader.readAsDataURL(file);
}

// File input
document.getElementById('imageInput').addEventListener('change', e => {
  const f = e.target.files[0]; if (!f) return; loadImageFile(f);
});

// OCR
document.getElementById('doOCR').addEventListener('click', async () => {
  if (!originalImage) return alert('Load an image first');
  const dataURL = canvas.toDataURL({format:'png'});
  const out = document.getElementById('ocrBoxes'); out.innerHTML='Detecting...';

  const worker = Tesseract.createWorker({ logger: m => console.log(m) });
  await worker.load();
  await worker.loadLanguage('eng');
  await worker.initialize('eng');
  const { data } = await worker.recognize(dataURL);
  await worker.terminate();

  ocrResults = data.words.map(w => ({ text: w.text, bbox: w.bbox }));
  out.innerHTML = '';
  ocrResults.forEach((w, i) => {
    const rect = new fabric.Rect({
      left: w.bbox.x0, top: w.bbox.y0,
      width: w.bbox.x1 - w.bbox.x0, height: w.bbox.y1 - w.bbox.y0,
      fill: 'rgba(255,0,0,0.15)', selectable:true, stroke:'red', strokeWidth:1, opacity:0.6
    });
    rect.set('data-ocr-index', i);
    canvas.add(rect);

    const label = document.createElement('div');
    label.className='box';
    label.textContent = `${i}: "${w.text}"`;
    out.appendChild(label);
  });
  canvas.renderAll();
});

// Replace selected text
function sampleBackgroundColor(x, y){
  const ctx = canvas.lowerCanvasEl.getContext('2d');
  try{
    const imgData = ctx.getImageData(Math.max(0,x), Math.max(0,y), 1, 1).data;
    return `rgb(${imgData[0]},${imgData[1]},${imgData[2]})`;
  }catch(e){
    return '#ffffff';
  }
}

document.getElementById('replaceSelectedText').addEventListener('click', () => {
  const repl = document.getElementById('replaceTextInput').value || '';
  const sel = canvas.getActiveObjects();
  if (!sel || sel.length===0) return alert('Select the OCR box(es) you want to replace');

  sel.forEach(obj => {
    const idx = obj.get('data-ocr-index');
    if (idx === undefined) return;
    const w = ocrResults[idx];
    const x = obj.left, y = obj.top, width = obj.width*obj.scaleX, height = obj.height*obj.scaleY;

    const bg = sampleBackgroundColor(Math.round(x), Math.round(y));

    const cover = new fabric.Rect({ left: x, top: y, width, height, fill: bg, selectable:false });
    const text = new fabric.Text(repl, { left: x + 4, top: y + (height/2)-10, fontSize: Math.max(10, Math.min(36, height*0.6)), selectable:false, fill:'#000' });
    canvas.add(cover);
    canvas.add(text);
    canvas.remove(obj);
  });
  canvas.discardActiveObject();
  canvas.renderAll();
});

// Image replacement region
let replacementImageFile = null;
document.getElementById('replacementFile').addEventListener('change', e => { replacementImageFile = e.target.files[0]; });

let drawingRect = false; let rect, startX, startY;
canvas.on('mouse:down', function(o){
  if (!originalImage) return;
  if (!replacementImageFile) return;
  drawingRect = true;
  const pointer = canvas.getPointer(o.e);
  startX = pointer.x; startY = pointer.y;
  rect = new fabric.Rect({ left:startX, top:startY, width:0, height:0, fill:'rgba(0,0,255,0.15)', stroke:'blue', strokeWidth:1, selectable:true });
  canvas.add(rect);
});

canvas.on('mouse:move', function(o){
  if (!drawingRect || !rect) return;
  const pointer = canvas.getPointer(o.e);
  rect.set({ width: Math.abs(pointer.x - startX), height: Math.abs(pointer.y - startY), left: Math.min(pointer.x, startX), top: Math.min(pointer.y, startY) });
  canvas.renderAll();
});

canvas.on('mouse:up', function(){ drawingRect = false; });

document.getElementById('applyReplacementImage').addEventListener('click', () => {
  if (!replacementImageFile) return alert('Choose replacement image file first');
  const active = canvas.getActiveObject();
  if (!active || !(active instanceof fabric.Rect)) return alert('Select the rectangle to replace');

  const reader = new FileReader();
  reader.onload = function(e){
  document.getElementById("originalPreview").src = e.target.result;
    fabric.Image.fromURL(e.target.result, img => {
      img.set({ left: active.left, top: active.top, selectable:false });
      const scaleX = active.width / img.width * active.scaleX;
      const scaleY = active.height / img.height * active.scaleY;
      img.scale(Math.max(scaleX, scaleY));
      canvas.add(img);
      canvas.remove(active);
      canvas.renderAll();
    });
  }
  reader.readAsDataURL(replacementImageFile);
});

document.getElementById('saveServer').addEventListener('click', async () => {
  const dataURL = canvas.toDataURL({format:'png'});
  const resp = await fetch('/save-image', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ imageData: dataURL }) });
  const j = await resp.json();
  if (j.ok) alert('Saved as: ' + j.filename);
  else alert('Save failed');
});


// Download original
document.getElementById('downloadOriginal').addEventListener('click', () => {
  const src = document.getElementById('originalPreview').src;
  if (!src) return alert('No original image loaded');
  const a = document.createElement('a');
  a.href = src;
  a.download = 'original.png';
  a.click();
});

// Download modified
document.getElementById('downloadModified').addEventListener('click', () => {
  const dataURL = canvas.toDataURL({format:'png'});
  const a = document.createElement('a');
  a.href = dataURL;
  a.download = 'modified.png';
  a.click();
});
