
// Improved OCR Version with Upscale + Malay Support + Contrast Boost

const canvas = new fabric.Canvas('c', { selection: true, preserveObjectStacking: true });
let originalImage = null;
let ocrResults = [];

// Upscale & Enhance for OCR
function upscaleForOCR(dataURL) {
  return new Promise(resolve => {
    const img = new Image();
    img.onload = () => {
      const cv = document.createElement("canvas");
      const ctx = cv.getContext("2d");
      cv.width = img.width * 2;
      cv.height = img.height * 2;
      ctx.drawImage(img, 0, 0, cv.width, cv.height);
      resolve(cv.toDataURL("image/png"));
    };
    img.src = dataURL;
  });
}

function enhanceForOCR(dataURL) {
  return new Promise(resolve => {
    const img = new Image();
    img.onload = () => {
      const cv = document.createElement("canvas");
      const ctx = cv.getContext("2d");
      cv.width = img.width;
      cv.height = img.height;
      ctx.drawImage(img, 0, 0);

      const imageData = ctx.getImageData(0, 0, cv.width, cv.height);
      const d = imageData.data;

      for (let i = 0; i < d.length; i += 4) {
        const avg = (d[i] + d[i+1] + d[i+2]) / 3;
        const bw = avg < 140 ? 0 : 255;
        d[i] = d[i+1] = d[i+2] = bw;
      }

      ctx.putImageData(imageData, 0, 0);
      resolve(cv.toDataURL("image/png"));
    };
    img.src = dataURL;
  });
}

// Load Image
function loadImageFile(file){
  const reader = new FileReader();
  reader.onload = function(e){
    document.getElementById("originalPreview").src = e.target.result;

    fabric.Image.fromURL(e.target.result, img => {
      const scale = Math.min(canvas.width / img.width, canvas.height / img.height, 1);
      img.set({ left: 0, top: 0, selectable: false, hasControls:false });
      img.scale(scale);

      canvas.clear();
      canvas.setBackgroundImage(img, canvas.renderAll.bind(canvas));
      originalImage = img;
      ocrResults = [];
      document.getElementById('ocrBoxes').innerHTML = '';
    });
  }
  reader.readAsDataURL(file);
}

document.getElementById('imageInput').addEventListener('change', e => {
  const f = e.target.files[0]; if (!f) return; loadImageFile(f);
});

// OCR Improved
document.getElementById('doOCR').addEventListener('click', async () => {
  if (!originalImage) return alert('Load an image first');

  const out = document.getElementById('ocrBoxes');
  out.innerHTML='Detecting...';

  let dataURL = canvas.toDataURL({format:'png'});
  const upscaled = await upscaleForOCR(dataURL);
  const enhanced = await enhanceForOCR(upscaled);

  const worker = await Tesseract.createWorker({
    corePath: "https://unpkg.com/tesseract.js-core@v4.0.0-simd/tesseract-core-simd.wasm.js",
    workerPath: "https://unpkg.com/tesseract.js@4.1.1/dist/worker.min.js",
    logger: m => console.log(m)
  });

  await worker.load();
  await worker.loadLanguage('eng+msa');
  await worker.initialize('eng+msa');

  const { data } = await worker.recognize(enhanced);
  await worker.terminate();

  ocrResults = data.lines; // Improved vs words[]
  out.innerHTML = '';

  ocrResults.forEach((w, i) => {
    const rect = new fabric.Rect({
      left: w.bbox.x0,
      top: w.bbox.y0,
      width: w.bbox.x1 - w.bbox.x0,
      height: w.bbox.y1 - w.bbox.y0,
      fill: 'rgba(0,255,0,0.15)',
      selectable:true,
      stroke:'green',
      strokeWidth:1
    });
    rect.set('data-ocr-index', i);
    canvas.add(rect);

    const label = document.createElement('div');
    label.className='box';
    label.textContent = `${i}: "${w.text}"`;
    out.appendChild(label);
  });

  canvas.renderAll();
});

// ==== Rest of functionality (replace text, image region, download buttons) ====

// Replace selected text
function sampleBackgroundColor(x, y){
  const ctx = canvas.lowerCanvasEl.getContext('2d');
  try{
    const imgData = ctx.getImageData(Math.max(0,x), Math.max(0,y), 1, 1).data;
    return `rgb(${imgData[0]},${imgData[1]},${imgData[2]})`;
  }catch(e){
    return '#ffffff';
  }
}

document.getElementById('replaceSelectedText').addEventListener('click', () => {
  const repl = document.getElementById('replaceTextInput').value || '';
  const sel = canvas.getActiveObjects();
  if (!sel.length) return alert('Select the OCR box');

  sel.forEach(obj => {
    const idx = obj.get('data-ocr-index');
    if (idx === undefined) return;

    const x = obj.left, y = obj.top, w = obj.width*obj.scaleX, h = obj.height*obj.scaleY;
    const bg = sampleBackgroundColor(Math.round(x), Math.round(y));

    canvas.add(new fabric.Rect({ left:x, top:y, width:w, height:h, fill:bg, selectable:false }));
    canvas.add(new fabric.Text(repl, { left:x+4, top:y+(h/2)-10, fontSize:h*0.5, selectable:false }));
    canvas.remove(obj);
  });

  canvas.discardActiveObject();
  canvas.renderAll();
});

// Replacement region
let replacementImageFile = null;
document.getElementById('replacementFile').addEventListener('change', e => { replacementImageFile = e.target.files[0]; });

let drawingRect = false, rect, startX, startY;
canvas.on('mouse:down', function(o){
  if (!originalImage || !replacementImageFile) return;
  drawingRect = true;
  const p = canvas.getPointer(o.e);
  startX = p.x; startY = p.y;
  rect = new fabric.Rect({ left:startX, top:startY, width:0, height:0, fill:'rgba(0,0,255,0.15)' });
  canvas.add(rect);
});
canvas.on('mouse:move', function(o){
  if (!drawingRect) return;
  const p = canvas.getPointer(o.e);
  rect.set({ width:Math.abs(p.x-startX), height:Math.abs(p.y-startY), left:Math.min(p.x,startX), top:Math.min(p.y,startY) });
  canvas.renderAll();
});
canvas.on('mouse:up', ()=> drawingRect=false);

document.getElementById('applyReplacementImage').addEventListener('click', () => {
  const active = canvas.getActiveObject();
  if (!active) return alert('Select rectangle');

  const reader = new FileReader();
  reader.onload = function(e){
    fabric.Image.fromURL(e.target.result, img => {
      img.set({ left:active.left, top:active.top, selectable:false });
      img.scale(Math.max(active.width/img.width, active.height/img.height));
      canvas.add(img);
      canvas.remove(active);
      canvas.renderAll();
    });
  }
  reader.readAsDataURL(replacementImageFile);
});

// Download buttons
document.getElementById('downloadOriginal').addEventListener('click', () => {
  const src = document.getElementById('originalPreview').src;
  if (!src) return alert('No original image loaded');
  const a = document.createElement('a');
  a.href = src;
  a.download = 'original.png';
  a.click();
});

document.getElementById('downloadModified').addEventListener('click', () => {
  const url = canvas.toDataURL("image/png");
  const a = document.createElement('a');
  a.href = url;
  a.download = 'modified.png';
  a.click();
});

// Save to server
document.getElementById('saveServer').addEventListener('click', async () => {
  const dataURL = canvas.toDataURL("image/png");
  const resp = await fetch('/save-image', {
    method:'POST',
    headers:{'Content-Type':'application/json'},
    body:JSON.stringify({ imageData:dataURL })
  });
  const j = await resp.json();
  if (j.ok) alert('Saved: ' + j.filename);
});
