// KawduLive — Video Chat App (React + WebRTC)
import React, { useEffect, useRef, useState } from "react";
import { io } from "socket.io-client";

const SIGNALING_SERVER = import.meta.env.VITE_SIGNALING || "http://localhost:4000";
const STUN_SERVERS = [{ urls: "stun:stun.l.google.com:19302" }];

export default function App() {
  const localVideoRef = useRef(null);
  const remoteVideoRef = useRef(null);
  const socketRef = useRef(null);
  const pcRef = useRef(null);
  const roomRef = useRef(null);

  const [status, setStatus] = useState("disconnected");
  const [muted, setMuted] = useState(false);

  useEffect(() => {
    socketRef.current = io(SIGNALING_SERVER);
    const socket = socketRef.current;

    socket.on("connect", () => setStatus("connected"));
    socket.on("waiting", () => setStatus("Waiting for KawduLive partner..."));

    socket.on("matched", async ({ room, initiator }) => {
      roomRef.current = room;
      setStatus("Matched — Connecting…");
      await startLocalStream();

      createPeerConnection(initiator);

      if (initiator) {
        const offer = await pcRef.current.createOffer();
        await pcRef.current.setLocalDescription(offer);
        socket.emit("signal", { room, data: { type: "sdp", sdp: offer } });
      }
    });

    socket.on("signal", async ({ data }) => {
      if (!pcRef.current) createPeerConnection(false);

      if (data.type === "sdp") {
        await pcRef.current.setRemoteDescription(new RTCSessionDescription(data.sdp));

        if (data.sdp.type === "offer") {
          const answer = await pcRef.current.createAnswer();
          await pcRef.current.setLocalDescription(answer);
          socketRef.current.emit("signal", {
            room: roomRef.current,
            data: { type: "sdp", sdp: answer },
          });
        }
      } else if (data.type === "ice") {
        try {
          await pcRef.current.addIceCandidate(new RTCIceCandidate(data.candidate));
        } catch (e) {
          console.warn("KawduLive ICE error", e);
        }
      }
    });

    socket.on("peer_left", () => {
      setStatus("Partner left");
      closeConnection();
    });

    return () => {
      socket.disconnect();
      closeConnection();
    };
  }, []);

  async function startLocalStream() {
    try {
      const stream = await navigator.mediaDevices.getUserMedia({
        video: true,
        audio: true
      });
      localVideoRef.current.srcObject = stream;
    } catch (err) {
      alert("KawduLive cannot access camera or microphone.");
    }
  }

  function createPeerConnection(initiator) {
    pcRef.current = new RTCPeerConnection({ iceServers: STUN_SERVERS });

    const localStream = localVideoRef.current.srcObject;
    localStream.getTracks().forEach(t => pcRef.current.addTrack(t, localStream));

    pcRef.current.ontrack = event => {
      remoteVideoRef.current.srcObject = event.streams[0];
    };

    pcRef.current.onicecandidate = event => {
      if (event.candidate) {
        socketRef.current.emit("signal", {
          room: roomRef.current,
          data: { type: "ice", candidate: event.candidate }
        });
      }
    };

    pcRef.current.onconnectionstatechange = () => {
      if (pcRef.current.connectionState === "connected") {
        setStatus("Connected on KawduLive");
      }
      if (
        pcRef.current.connectionState === "failed" ||
        pcRef.current.connectionState === "disconnected"
      ) {
        setStatus("Connection ended");
      }
    };
  }

  function closeConnection() {
    try {
      pcRef.current && pcRef.current.close();
    } catch {}

    pcRef.current = null;

    let vid1 = localVideoRef.current?.srcObject;
    let vid2 = remoteVideoRef.current?.srcObject;

    vid1 && vid1.getTracks().forEach(t => t.stop());
    vid2 && vid2.getTracks().forEach(t => t.stop());

    if (localVideoRef.current) localVideoRef.current.srcObject = null;
    if (remoteVideoRef.current) remoteVideoRef.current.srcObject = null;

    roomRef.current = null;
  }

  function findMatch() {
    socketRef.current.emit("find_match");
    setStatus("Searching for KawduLive partner...");
  }

  function leaveCall() {
    socketRef.current.emit("leave", { room: roomRef.current });
    closeConnection();
    setStatus("Left call");
  }

  function toggleMute() {
    const stream = localVideoRef.current?.srcObject;
    if (!stream) return;

    stream.getAudioTracks().forEach(t => (t.enabled = !t.enabled));
    setMuted(!muted);
  }

  return (
    <div style={{ padding: 20, fontFamily: "Arial" }}>
      <h1>KawduLive</h1>
      <p>{status}</p>

      <div style={{ display: "flex", gap: "16px" }}>
        <div>
          <h4>You</h4>
          <video
            ref={localVideoRef}
            autoPlay
            playsInline
            muted
            style={{ width: 300, background: "#000" }}
          />
        </div>

        <div>
          <h4>Partner</h4>
          <video
            ref={remoteVideoRef}
            autoPlay
            playsInline
            style={{ width: 300, background: "#000" }}
          />
        </div>
      </div>

      <div style={{ marginTop: 20 }}>
        <button onClick={findMatch}>Find Partner</button>
        <button onClick={leaveCall} style={{ marginLeft: 8 }}>
          Leave
        </button>
        <button onClick={toggleMute} style={{ marginLeft: 8 }}>
          {muted ? "Unmute" : "Mute"}
        </button>
      </div>

      <p style={{ marginTop: 10, fontSize: 12 }}>
        KawduLive uses STUN only. Add TURN for real public usage.
      </p>
    </div>
  );
}
